/*
 * Isomorphic SmartGWT web presentation layer
 * Copyright 2000 and beyond Isomorphic Software, Inc.
 *
 * OWNERSHIP NOTICE
 * Isomorphic Software owns and reserves all rights not expressly granted in this source code,
 * including all intellectual property rights to the structure, sequence, and format of this code
 * and to all designs, interfaces, algorithms, schema, protocols, and inventions expressed herein.
 *
 *  If you have any questions, please email <sourcecode@isomorphic.com>.
 *
 *  This entire comment must accompany any portion of Isomorphic Software source code that is
 *  copied or moved from this file.
 */

package com.smartgwt.sample.showcase.client.shuttles;

import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.client.widgets.form.DynamicForm;
import com.smartgwt.client.widgets.form.fields.TextItem;
import com.smartgwt.client.widgets.form.fields.ShuttleItem;
import com.smartgwt.client.widgets.form.fields.SubmitItem;
import com.smartgwt.client.widgets.grid.ListGrid;
import com.smartgwt.client.widgets.grid.ListGridRecord;
import com.smartgwt.client.widgets.grid.events.SelectionUpdatedEvent;
import com.smartgwt.client.widgets.grid.events.SelectionUpdatedHandler;
import com.smartgwt.client.types.SelectionStyle;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;
import com.smartgwt.sample.showcase.client.SourceEntity;

public class DataBoundShuttleSample extends ShowcasePanel {
    private static final String DESCRIPTION = 
        " This example demonstrates using a ShuttleItem to edit a many-to-many relationship "+
        "between two sets of data - Employees and Teams. Each Team can have multiple Employees"+
        "as members, and each Employee may be in multiple teams."+
        "<p>"+
        "Select a Team from the grid at the top to edit it in the form below. The ShuttleItem allows "+
        "you to add employees to the team and remove them using a familiar shuttle style interface, "+
        "and then save the changes."+
        "<p>"+
        "The data for this sample is stored in three database tables - an Employee table, a Teams table "+
        "and a pivot table with entries connecting Employees to Teams. Each database table has an "+
        "associated DataSource."+
        "<p>"+
        "The Teams dataSource contains an EmployeeId field with multiple set to true and a foreignKey "+
        "declaration which maps through the dataSource for the pivot table to the related Employees "+
        "dataSource. The Employees DataSource has a similar TeamId field with the reciprocal mapping "+
        "through the pivot table back to the Teams dataSource."+
        "<p>"+
        "On fetch, the SmartClient server will dynamically populate these field with an appropriate array "+
        "of values from the entries in the pivot table, so each Team record will have EmployeeId set to "+
        "the array of TeamIds the employee is associated with."+
        "<p>"+
        "When the user modifies these values and saves the changes, the SmartClient server will update the "+
        "pivot table to reflect the new set of employees for the team.";
        
    public static class Factory implements PanelFactory {

        private String id;

        public ShowcasePanel create() {
            DataBoundShuttleSample panel = new DataBoundShuttleSample();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }
    
    protected boolean isTopIntro() {
        return true;
    }

    public Canvas getViewPanel() {
        DataSource ds = DataSource.get("teams_relation");
        
        DynamicForm teamForm = new DynamicForm();
        teamForm.setDataSource(ds);
        teamForm.disable();
    	
        TextItem nameItem = new TextItem("TeamName");
        ShuttleItem employeeIdItem = new ShuttleItem("EmployeeId");
        employeeIdItem.setCanEdit(true);
        employeeIdItem.setDisplayField("Name");
        SubmitItem submitButtonItem = new SubmitItem();
        submitButtonItem.setTitle("Save");
    	
        teamForm.setFields(nameItem,employeeIdItem,submitButtonItem);
    	
        ListGrid teamGrid = new ListGrid();
        teamGrid.setDataSource(ds);
        teamGrid.setAutoFetchData(true);
        teamGrid.setHeight(200);
        teamGrid.setShowResizeBar(true);
        teamGrid.setSelectionType(SelectionStyle.SINGLE);
        teamGrid.addSelectionUpdatedHandler(new SelectionUpdatedHandler() {
            @Override
            public void onSelectionUpdated(SelectionUpdatedEvent event) {
                ListGridRecord rec = teamGrid.getSelectedRecord();
                if (rec == null) {
                    teamForm.clearValues();
                    teamForm.disable();
                } else {
                    teamForm.enable();
                    teamForm.editRecord(rec);
                }
            }
        });
    	
        VLayout vLayout = new VLayout();
        vLayout.setWidth100();
        vLayout.setHeight100();
        vLayout.setMembers(teamGrid, teamForm);
        
        return vLayout;
    }

    public String getIntro() {
        return DESCRIPTION;
    }

    @Override
    public SourceEntity[] getSourceUrls() {
        return new SourceEntity[]{
            new SourceEntity("teamMembers_relation.ds.xml", XML, "source/teamMembers_relation.ds.xml.html", true),
            new SourceEntity("employees_relation.ds.xml", XML, "source/employees_relation.ds.xml.html", true)
        };
    }
    
}
