package com.smartgwt.sample.showcase.client.grid.filtering;

import com.google.gwt.user.client.Timer;
import com.smartgwt.client.data.DSRequest;
import com.smartgwt.client.data.DSResponse;
import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.data.Hilite;
import com.smartgwt.client.data.ResultSet;
import com.smartgwt.client.data.Criterion;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.data.ResponseTransformer;
import com.smartgwt.client.widgets.IButton;
import com.smartgwt.client.types.ListGridFieldType;
import com.smartgwt.client.types.OperatorId;
import com.smartgwt.client.types.DSOperationType;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.Label;
import com.smartgwt.client.widgets.grid.ListGrid;
import com.smartgwt.client.widgets.grid.ListGridField;
import com.smartgwt.client.widgets.events.ClickEvent;
import com.smartgwt.client.widgets.events.ClickHandler;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.client.widgets.layout.HLayout;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;
import com.smartgwt.sample.showcase.client.SourceEntity;

public class SpreadsheetFilteringSample extends ShowcasePanel {

    private static final String DESCRIPTION = "The SetFilterItem offers a familiar spreadsheet-style "+
        "filtering experience. "+
        "Users can filter each column by the values currently present in the column."+
        "<P>"+
        "This example demonstrates three different ways to utilize a SetFilterItem within a "+
        "ListGrid's filter editor."+
        "<ul><li>The \"Item\" field uses a simple SetFilterItem without special configuration."+
        "    In this standard usage, the distinct values from the target field are automatically "+
        "    displayed in the picker, with duplicate values removed. Where possible, the list of "+
        "    filter options is derived directly from the the data in the list. "+
        "    If the list does not contain a complete cache of data, the SetFilteritem will "+
        "    automatically issue a query to the list's DataSource to retrieve the distinct "+
        "    values from the complete data set.</li>"+
        "<li>The \"Category\" field is configured to pull its filter options from a related "+
        "    optionDatasource instead of extracting them from the list.</li>"+
        "<li>The \"Units\" field, which is an enum type, demonstrates how the SetFilterItem "+
        "    can also handle fields with a statically defined valueMap of valid values.</li>"+
        "</ul>"+
        "Try the following actions:"+
        "<ol><li>Show the Item field filter drop down. Since the grid does not have a full set of "+
        "    data loaded by default, a second fetch will be issued against the <code>supplyItem</code> "+ 
        "    dataSource to pick up the list of unique itemNames. "+
        "    Now recreate the grid, hit \"Fetch Unloaded Records\" and try again. "+
        "    Now the set of options is able to be derived directly from the already loaded "+
        "    data in the grid.</li>"+
        "<li>Show the Category filter drop down. For this field, the Category options are "+
        "    retrieved from the related <code>supplyCategory</code> dataSource. "+
        "    This SetFilterItem is also configured to show additional <code>supplyCategory</code> fields. "+
        "    Hit the expansion icon within the drop down to enter an expanded view showing "+
        "    Parent Category as well as Category.</li>"+
        "<li>Unlike the \"Item\" and \"Category\" fields, the \"Units\" filter options are derived "+
        "    from a defined valueMap. Showing the filter drop down for this field will never "+
        "    issue a server fetch.</li></ol>";

    public static class Factory implements PanelFactory {
        private String id;

        public ShowcasePanel create() {
            SpreadsheetFilteringSample panel = new SpreadsheetFilteringSample();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }

    protected boolean isTopIntro() {
        return true;
    }
    
    private Label serverCountLabel;
    private ListGrid supplyGrid;
    private HLayout mainLayout;
    private VLayout leftVLayout;
    
    private class ServerCountLabel extends Label {
        private int count = 0;  
  
        public ServerCountLabel() {  
            setWidth100();
            setHeight(40);
            setPadding(10);
            setBorder("1px solid grey");
            setContents("<b>Number of server trips: 0</b>");
        }  
  
        public void updateContent (String dataSource, int totalRows, int startRow, int endRow) {
            count++;
            serverCountLabel.setContents("Number of server trips: " + this.count + 
                                         "<br><br>DataSource: \"" + dataSource  + "\"" +
                                         "<br/>Total rows in this filter set: " + totalRows + 
                                         "<br/>Last range of records returned: " + startRow + " to " + endRow );
        }
    } 
    
    private DataSource supplyItemDS = DataSource.get("supplyItem", null, new ResponseTransformer() {
        protected void transformResponse(DSResponse response, DSRequest request, Object data) {
            if(request.getOperationType() == DSOperationType.FETCH) {
                int totalRows = response.getTotalRows(); 
                int startRow = response.getStartRow(); 
                int endRow = response.getEndRow(); 
                ((ServerCountLabel) serverCountLabel).updateContent(supplyItemDS.getID(), totalRows, startRow, endRow); 
                serverCountLabel.setBackgroundColor("#ffff77"); 
                new Timer() { 
                    public void run() { 
                        serverCountLabel.setBackgroundColor("#ffffff"); 
                    } 
                }.schedule(500); 
            }
            defaultTransformResponse(response, request, data);
        }
    });
    
    public Canvas getViewPanel() {
        
        leftVLayout = new VLayout();
        leftVLayout.setMembersMargin(20);
        
        mainLayout = new HLayout();
        mainLayout.setWidth100();
        mainLayout.setHeight100();        
        mainLayout.setMembersMargin(20);
    	
        VLayout rightVLayout = new VLayout();
        rightVLayout.setMembersMargin(20);
        
        IButton fetchUnloadedRecordsButton = new IButton();
        fetchUnloadedRecordsButton.setWidth(200);
        fetchUnloadedRecordsButton.setTitle("Fetch Unloaded Records");
        fetchUnloadedRecordsButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                ResultSet rest = supplyGrid.getResultSet();
                if (!rest.rangeIsLoaded(0, rest.getLength())) {
                    // getRange will fetch the requested rows from the server
                    rest.getRange(0, rest.getLength());
                }
            }
        });
        IButton recreateGridButton = new IButton();
        recreateGridButton.setWidth(200);
        recreateGridButton.setTitle("Recreate Grid");
        recreateGridButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                (leftVLayout.getMember(0)).destroy();
                createGrid();
            }
        });
        
        serverCountLabel = new ServerCountLabel();
        
        leftVLayout.setMembers(serverCountLabel);
        rightVLayout.setMembers(fetchUnloadedRecordsButton, recreateGridButton);
    	
        mainLayout.setMembers(leftVLayout, rightVLayout);
        
        createGrid();

        return mainLayout;
    }

    private void createGrid () {
        supplyGrid = new ListGrid();    	
        supplyGrid.setWidth100();
        supplyGrid.setHeight(300);
        supplyGrid.setShowFilterEditor(true);
        supplyGrid.setFilterOnKeypress(true);
        supplyGrid.setFetchDelay(500);
        supplyGrid.setAutoFetchData(true);
        supplyGrid.setDataSource(DataSource.get("supplyItem"));
    	
        ListGridField SKU = new ListGridField("SKU");
        ListGridField itemName = new ListGridField("itemName");
        ListGridField category = new ListGridField("category");
        ListGridField units = new ListGridField("units");
    	
        supplyGrid.setFields(SKU,itemName,category,units);
    	
        leftVLayout.addMember(supplyGrid, 0);
    }
    
    public String getIntro() {
        return DESCRIPTION;
    }
    
    @Override
    public SourceEntity[] getSourceUrls() {
        return new SourceEntity[]{
            new SourceEntity("supplyCategory.ds.xml", XML, "source/supplyCategory.ds.xml.html", true)
        };
    }
    
}