package com.smartgwt.sample.showcase.client.dataintegration.java;

import java.util.HashMap;
import java.util.Map;

import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.data.AdvancedCriteria;
import com.smartgwt.client.data.Criterion;
import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.types.FetchMode;
import com.smartgwt.client.types.OperatorId;
import com.smartgwt.client.util.Authentication;
import com.smartgwt.client.widgets.IButton;
import com.smartgwt.client.widgets.events.ClickEvent;
import com.smartgwt.client.widgets.events.ClickHandler;
import com.smartgwt.client.widgets.form.DynamicForm;
import com.smartgwt.client.widgets.form.fields.SelectItem;
import com.smartgwt.client.widgets.form.fields.events.ChangedEvent;
import com.smartgwt.client.widgets.form.fields.events.ChangedHandler;
import com.smartgwt.client.widgets.grid.ListGridField;
import com.smartgwt.client.widgets.layout.HLayout;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.client.widgets.tree.TreeGrid;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;

public class DeclarativeSecuritySample extends ShowcasePanel {

    private static final String DESCRIPTION =
        "<p>The <i>Declarative Security</i> system lets you control which users can access what data "+
        "and what operations, by simply declaring the roles that are allowed access."+
        "<p>"+
        "This security system is <i>pluggable</i>, meaning that it works with <i>any</i> authentication "+
        "system: you just tell our framework what roles the current user has, and we enforce the rules."+
        "<p>"+
        "The sample below is a client-only sample which <b>simulates</b> the server-side Declarative "+
        "Security system - we did it this way so you could try out different roles in the sample. However, "+
        "this simulator is extremely valuable as well, because it allows you to "+
        "implement and test the effect of roles on the UI, and then switch over to the real authentication "+
        "system without changing any of the UI code."+
        "<p>"+
        "In the sample below, note how you cannot edit anything and salaries are not displayed. If you give "+
        "yourself the HR role, you will notice that a \"Salary\" field appears, and you are now able to edit "+
        "employee records."+
        "<p>"+
        "With the CEO role, a \"Remove Employee\" button appears, which you can use to remove an employee. "+
        "This button is configured to appear only if you have the CEO role."+
        "<p>"+
        "There's a second button, \"Cheater Remove Employee\", which is always present, but doesn't work "+
        "unless you are actually the CEO. This shows that the rules are being enforced even if someone makes "+
        "a mistake and the UI offers options that a user should not have.";
    
    public static class Factory implements PanelFactory {
        private String id;

        public ShowcasePanel create() {
            DeclarativeSecuritySample panel = new DeclarativeSecuritySample();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }

    protected boolean isTopIntro() {
        return true;
    }

    public Canvas getViewPanel() {
        
        // we need to call setCurrentUser(), otherwise the client security code assumes you are not authenticated.
        Record user = new Record();
        user.setAttribute("userId", "jean");
        Authentication.setCurrentUser(user);
		
        final VLayout mainLayout = new VLayout();
        mainLayout.setMembersMargin(15);
		
        DynamicForm formRoles = new DynamicForm();
		
        SelectItem selectRoles = new SelectItem();
        selectRoles.setName("roles");
        selectRoles.setTitle("Roles");
        selectRoles.setMultiple(true);
        Map<String,String> valuesMap = new HashMap<String,String>();
        valuesMap.put("HR", "HR");
        valuesMap.put("CEO", "CEO");
        selectRoles.setValueMap(valuesMap);
        selectRoles.addChangedHandler(new ChangedHandler() {
            @Override
            public void onChanged(ChangedEvent event) {
                // setRoles() is meant to be called before application startup, and the roles should be provided by whatever system
                // you use for authentication. Note that you are *not* normally allowed to change roles after your application has started;
                // this sample simulates dynamic role change by completely rebuilding the UI.
                Object values = event.getForm().getValuesAsRecord().getAttribute("roles");
                String[] roles = null;
                if (values != null) roles = ((String)values).split(",");

                Authentication.setRoles(roles); 
                ((HLayout)mainLayout.getMember(1)).destroy();
                mainLayout.addMember(createUI());
            }
        });
		
        formRoles.setFields(selectRoles);
		
        mainLayout.addMember(formRoles);
        mainLayout.addMember(createUI());

        return mainLayout;
    }

    private HLayout createUI() {
        final TreeGrid employeeTree = new TreeGrid();
        employeeTree.setID("employeeTree");
        employeeTree.setWidth(525);
        employeeTree.setHeight(400);
        employeeTree.setDataSource(DataSource.get("employeesCO"));
        employeeTree.setNodeIcon("icons/16/person.png");
        employeeTree.setFolderIcon("icons/16/person.png");
        employeeTree.setShowOpenIcons(false);
        employeeTree.setShowDropIcons(false);
        employeeTree.setClosedIconSuffix("");
        employeeTree.setAutoFetchData(true);
        employeeTree.setDataFetchMode(FetchMode.LOCAL);
        employeeTree.setLoadDataOnDemand(false);
        employeeTree.setShowSelectedIcons(true);
        employeeTree.setShowAllColumns(true);
        employeeTree.setAutoOpenTree("all");
		
        ListGridField nameField = new ListGridField();
        nameField.setTitle("Name");
        nameField.setWidth("40%");
        nameField.setName("name");
        ListGridField jobField = new ListGridField();
        jobField.setName("job");
        ListGridField salaryField = new ListGridField();
        salaryField.setName("salary");
		
        employeeTree.setFields(nameField,jobField,salaryField);
		
        AdvancedCriteria adCriteria = new AdvancedCriteria(OperatorId.OR, new Criterion[]{
                new Criterion("auth.roles", OperatorId.CONTAINS, "CEO"),
                new Criterion("auth.roles", OperatorId.CONTAINS, "HR")
            });
        employeeTree.addDynamicProperty("canEdit", adCriteria);

        VLayout buttonsLayout = new VLayout();
        buttonsLayout.setMembersMargin(15);
		
        IButton removeButton = new IButton();
        removeButton.setTitle("Remove Employee");
        removeButton.setWidth(200);
        AdvancedCriteria criteria1 = new AdvancedCriteria(OperatorId.AND, new Criterion[]{
                new Criterion("auth.roles", OperatorId.REGEXP, "CEO")
            });
        removeButton.setVisibleWhen(criteria1);
        AdvancedCriteria criteria2 = new AdvancedCriteria(OperatorId.AND, new Criterion[]{
                new Criterion("employeeTree.anySelected", OperatorId.EQUALS, true)
            });
        removeButton.setEnableWhen(criteria2);
        removeButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                employeeTree.removeData(employeeTree.getSelectedRecord());
            }
        });
		
        IButton cheaterRemoveButton = new IButton();
        cheaterRemoveButton.setTitle("Cheater Remove Employee");
        cheaterRemoveButton.setWidth(200);
        cheaterRemoveButton.setEnableWhen(criteria2);
        cheaterRemoveButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                employeeTree.removeData(employeeTree.getSelectedRecord());
            }
        });
		
        buttonsLayout.addMember(removeButton);
        buttonsLayout.addMember(cheaterRemoveButton);
		
        HLayout hLayout = new HLayout();
        hLayout.setMembersMargin(15);
        hLayout.addMember(employeeTree);
        hLayout.addMember(buttonsLayout);
		
        return hLayout;
	}


    public String getIntro() {
        return DESCRIPTION;
    }

}